<?php
namespace Piggly\Tests\Pix;

use PHPUnit\Framework\TestCase;
use Piggly\Pix\Parser;
use Piggly\Pix\Reader;
use Piggly\Pix\StaticPayload;
use Piggly\Pix\Utils\Cast;

/**
 * @coversDefaultClass \Piggly\Pix\StaticPayload
 */
class StaticPayloadCreationTest extends TestCase
{
	/**
	 * Assert if $actual is equals to $expected.
	 * 
	 * The main purpose here is not validate if
	 * pix can be processed by banks, but if it's
	 * generating as expected and being readable.
	 * 
	 * Anytime it runs will create 100 random pix
	 * codes. It must assert all anytime.
	 *
	 * @covers ::setPixKey
	 * @covers ::setDescription
	 * @covers ::setMerchantName
	 * @covers ::setMerchantCity
	 * @covers ::setPostalCode
	 * @covers ::setTid
	 * @covers ::setAmount
	 * @covers ::getPixCode
	 * @covers Parser::getKeyType
	 * @covers Reader::getAmount
	 * @covers Reader::getPixKey
	 * @covers Reader::getDescription
	 * @covers Reader::getTid
	 * @covers Reader::getMerchantName
	 * @covers Reader::getMerchantCity
	 * @covers Reader::getPostalCode
	 * @dataProvider dataPix
	 * @test Expecting positive assertion.
	 * @param string $type Pix data type.
	 * @param string|float $expected Expected value.
	 * @param string $pixCode Pix code.
	 * @return boolean
	 */
	public function isMatching ( string $type, $expected, string $pixCode )
	{ 
		$reader = new Reader($pixCode);
		$this->assertEquals($expected, $this->getData($reader, $type)); 
	}

	/**
	 * A list with random pix to validate.
	 * Provider to isMatching() method.
	 * Generated by fakerphp.
	 * @return array
	 */
	public function dataPix () : array
	{
		$arr = [];

		$faker = \Faker\Factory::create('pt_BR');

		for ( $i = 0; $i < 100; $i++ )
		{
			$pix     = new StaticPayload();
			$pixData = ['AMOUNT' => 0];

			// Amount
			if ( $faker->boolean() ) 
			{
				$pixData['AMOUNT'] = $faker->randomFloat(2);
				$pix->setAmount($pixData['AMOUNT']);
			}

			// Key
			$pixData = array_merge($pixData, $this->getRandomKey($faker));
			$pix->setPixKey($pixData['KEY_TYPE'], $pixData['KEY_VALUE']);

			// Description
			if ( $faker->boolean() )
			{ 
				$pixData['DESC'] = $faker->words(3, true); 
				$pix->setDescription($pixData['DESC']);
			}

			// Tid
			if ( $faker->boolean() )
			{ 
				$pixData['TID'] = $faker->regexify('[A-Z]{2}[0-4]{8}'); 
				$pix->setTid($pixData['TID']);
			}

			// Merchant data
			$pixData['MERCHANT_NAME'] = $faker->firstName().' '.$faker->lastName(); 
			$pixData['MERCHANT_CITY'] = $faker->city(); 
			$pix->setMerchantCity($pixData['MERCHANT_CITY'])->setMerchantName($pixData['MERCHANT_NAME']);

			// Description
			if ( $faker->boolean() )
			{ 
				$pixData['CEP'] = $faker->postcode(); 
				$pix->setPostalCode($pixData['CEP']);
			}

			// Pix Code
			$pixCode = $pix->getPixCode();

			$arr[] = ['AMOUNT', $pixData['AMOUNT'], $pixCode];
			$arr[] = ['KEY_TYPE', $pixData['KEY_TYPE'], $pixCode];
			$arr[] = ['KEY_VALUE', Parser::parse($pixData['KEY_TYPE'], $pixData['KEY_VALUE']), $pixCode];
			$arr[] = ['DESC', Cast::upperStr(Cast::cleanStr($pixData['DESC']??''), true), $pixCode];
			$arr[] = ['TID', $pixData['TID']??'***', $pixCode];
			$arr[] = ['MERCHANT_NAME', \substr(Cast::upperStr(Cast::cleanStr($pixData['MERCHANT_NAME'])), 0, 25), $pixCode];
			$arr[] = ['MERCHANT_CITY', \substr(Cast::upperStr(Cast::cleanStr($pixData['MERCHANT_CITY'])), 0, 15), $pixCode];
			$arr[] = ['CEP', Cast::upperStr(Cast::cleanStr($pixData['CEP']??''), true), $pixCode];
		}

		return $arr;
	}

	/**
	 * Get data by $type from $reader.
	 *
	 * @param Reader $reader
	 * @param string $type
	 * @return void
	 */
	private function getData ( Reader $reader, string $type )
	{
		switch ( $type )
		{
			case 'AMOUNT':
				return $reader->getAmount();
			case 'KEY_TYPE':
				return Parser::getKeyType($reader->getPixKey());
			case 'KEY_VALUE':
				return $reader->getPixKey();
			case 'DESC':
				return $reader->getDescription();
			case 'TID':
				return $reader->getTid();
			case 'MERCHANT_NAME':
				return $reader->getMerchantName();
			case 'MERCHANT_CITY':
				return $reader->getMerchantCity();
			case 'CEP':
				return $reader->getPostalCode();
		}
	}

	/**
	 * Get random pix key.
	 * 
	 * @param Faker $faker
	 * @return array
	 */
	protected function getRandomKey ( $faker ) : array 
	{
		$num = $faker->numberBetween(0, 4);

		switch ( $num )
		{
			case 0:
				return ['KEY_TYPE' => Parser::KEY_TYPE_DOCUMENT, 'KEY_VALUE' => $faker->cnpj()];
			case 1:
				return ['KEY_TYPE' => Parser::KEY_TYPE_DOCUMENT, 'KEY_VALUE' => $faker->cpf()];
			case 2:
				return ['KEY_TYPE' => Parser::KEY_TYPE_EMAIL, 'KEY_VALUE' => $faker->email()];
			case 3:
				return ['KEY_TYPE' => Parser::KEY_TYPE_PHONE, 'KEY_VALUE' => $faker->phoneNumber()];
			case 4:
				return ['KEY_TYPE' => Parser::KEY_TYPE_RANDOM, 'KEY_VALUE' => $this->genUuid()];
		}
	}

	/**
	 * Remove invalid chars.
	 *
	 * @param string $str
	 * @return string|null
	 */
	protected function replacesChar ( ?string $str ) : ?string
	{
		if ( \is_null($str) )
		{ return null; }

		$invalid = array("Á", "À", "Â", "Ä", "Ă", "Ā", "Ã", "Å", "Ą", "Æ", "Ć", "Ċ", "Ĉ", "Č", "Ç", "Ď", "Đ", "Ð", "É", "È", "Ė", "Ê", "Ë", "Ě", "Ē", "Ę", "Ə", "Ġ", "Ĝ", "Ğ", "Ģ", "á", "à", "â", "ä", "ă", "ā", "ã", "å", "ą", "æ", "ć", "ċ", "ĉ", "č", "ç", "ď", "đ", "ð", "é", "è", "ė", "ê", "ë", "ě", "ē", "ę", "ə", "ġ", "ĝ", "ğ", "ģ", "Ĥ", "Ħ", "Í", "Ì", "İ", "Î", "Ï", "Ī", "Į", "Ĳ", "Ĵ", "Ķ", "Ļ", "Ł", "Ń", "Ň", "Ñ", "Ņ", "Ó", "Ò", "Ô", "Ö", "Õ", "Ő", "Ø", "Ơ", "Œ", "ĥ", "ħ", "ı", "í", "ì", "î", "ï", "ī", "į", "ĳ", "ĵ", "ķ", "ļ", "ł", "ń", "ň", "ñ", "ņ", "ó", "ò", "ô", "ö", "õ", "ő", "ø", "ơ", "œ", "Ŕ", "Ř", "Ś", "Ŝ", "Š", "Ş", "Ť", "Ţ", "Þ", "Ú", "Ù", "Û", "Ü", "Ŭ", "Ū", "Ů", "Ų", "Ű", "Ư", "Ŵ", "Ý", "Ŷ", "Ÿ", "Ź", "Ż", "Ž", "ŕ", "ř", "ś", "ŝ", "š", "ş", "ß", "ť", "ţ", "þ", "ú", "ù", "û", "ü", "ŭ", "ū", "ů", "ų", "ű", "ư", "ŵ", "ý", "ŷ", "ÿ", "ź", "ż", "ž");
		$valid   = array("A", "A", "A", "A", "A", "A", "A", "A", "A", "AE", "C", "C", "C", "C", "C", "D", "D", "D", "E", "E", "E", "E", "E", "E", "E", "E", "G", "G", "G", "G", "G", "a", "a", "a", "a", "a", "a", "a", "a", "a", "ae", "c", "c", "c", "c", "c", "d", "d", "d", "e", "e", "e", "e", "e", "e", "e", "e", "g", "g", "g", "g", "g", "H", "H", "I", "I", "I", "I", "I", "I", "I", "IJ", "J", "K", "L", "L", "N", "N", "N", "N", "O", "O", "O", "O", "O", "O", "O", "O", "CE", "h", "h", "i", "i", "i", "i", "i", "i", "i", "ij", "j", "k", "l", "l", "n", "n", "n", "n", "o", "o", "o", "o", "o", "o", "o", "o", "o", "R", "R", "S", "S", "S", "S", "T", "T", "T", "U", "U", "U", "U", "U", "U", "U", "U", "U", "U", "W", "Y", "Y", "Y", "Z", "Z", "Z", "r", "r", "s", "s", "s", "s", "B", "t", "t", "b", "u", "u", "u", "u", "u", "u", "u", "u", "u", "u", "w", "y", "y", "y", "z", "z", "z");
		$str     = str_ireplace( $invalid, $valid, $str );
		$str     = preg_replace('/[^A-Za-z0-9\s\-]+/', '', $str);

		return $str;
	}
	
	/**
	 * Generate random uuidv4.
	 *
	 * @return void
	 */
	protected function genUuid ()
	{
		return sprintf( '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
			// 32 bits for "time_low"
			mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff ),

			// 16 bits for "time_mid"
			mt_rand( 0, 0xffff ),

			// 16 bits for "time_hi_and_version",
			// four most significant bits holds version number 4
			mt_rand( 0, 0x0fff ) | 0x4000,

			// 16 bits, 8 bits for "clk_seq_hi_res",
			// 8 bits for "clk_seq_low",
			// two most significant bits holds zero and one for variant DCE1.1
			mt_rand( 0, 0x3fff ) | 0x8000,

			// 48 bits for "node"
			mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff )
		);
	}
}